/* global describe it before ethers */

// Import necessary libraries and types
import {
  getSelectors,
  FacetCutAction,
  removeSelectors,
  findAddressPositionInFacets,
} from '../../scripts/libraries/diamond';
import {
  DiamondCutFacet,
  DiamondLoupeFacet,
  OwnershipFacet,
} from '../../types';

import {
  deployDiamond,
  DiamondAddress,
} from '../../scripts/deployTradingManagement';

import { ethers } from 'hardhat';

import { ContractReceipt } from 'ethers';
import { assert, expect } from 'chai';

// Begin the test suite for the Trading Manager Diamond
describe('35 - Trading Manager Diamond Test', async function () {
  // Define variables to hold facet instances and addresses
  let diamondCutFacet: DiamondCutFacet;
  let diamondLoupeFacet: DiamondLoupeFacet;
  let ownershipFacet: OwnershipFacet;
  let tradingManagementStorageFacet;
  let tradingManagementExecutorFacet;
  let test1FacetAddress: string;
  let test2FacetAddress: string;
  let diamondCutFacetAddress: string;
  let diamondLoupeFacetAddress: string;
  let ownershipFacetAddress: string;
  let tradingManagementStorageFacetAddress: string;
  let tradingManagementExecutorFacetAddress: string;
  let tx;
  let receipt: ContractReceipt;
  let result;
  const addresses: string[] = [];

  // Before all tests, deploy the Diamond and get instances of the core facets
  before(async function () {
    // Deploy the Diamond contract
    const tradingManagementAddress = await deployDiamond({
      saveInfo: false,
      showInfo: false,
    });
    // Get instances of the core facets using the Diamond's address
    diamondCutFacet = await ethers.getContractAt(
      'DiamondCutFacet',
      DiamondAddress,
    );
    diamondLoupeFacet = await ethers.getContractAt(
      'DiamondLoupeFacet',
      DiamondAddress,
    );
    ownershipFacet = await ethers.getContractAt(
      'OwnershipFacet',
      DiamondAddress,
    );
    tradingManagementStorageFacet = await ethers.getContractAt(
      'TradingManagementStorageFacet',
      DiamondAddress,
    );
    tradingManagementExecutorFacet = await ethers.getContractAt(
      'TradingManagementExecutorFacet',
      DiamondAddress,
    );

    // Map function selectors to facet addresses
    diamondCutFacetAddress = await diamondLoupeFacet.facetAddress('0x1f931c1c'); // diamondCut()
    diamondLoupeFacetAddress =
      await diamondLoupeFacet.facetAddress('0x7a0ed627'); // facets()
    ownershipFacetAddress = await diamondLoupeFacet.facetAddress('0xf2fde38b'); // transferOwnership()
    tradingManagementStorageFacetAddress =
      await diamondLoupeFacet.facetAddress('0xf05a7662'); // setBourseContractAddress()
    tradingManagementExecutorFacetAddress =
      await diamondLoupeFacet.facetAddress('0xe9ee92fc'); // purchaseAccessRightPAYG()

    addresses.push(
      diamondCutFacetAddress,
      diamondLoupeFacetAddress,
      ownershipFacetAddress,
      tradingManagementStorageFacetAddress,
      tradingManagementExecutorFacetAddress,
    );
  });

  it('should have 5 initial facets: DiamondCutFacet, DiamondLoupeFacet, OwnershipFacet and TradingManagementStorageFacet', async () => {
    const facetAddresses = await diamondLoupeFacet.facetAddresses();
    assert.equal(facetAddresses.length, 5);
    assert.sameMembers(facetAddresses, addresses);
  });

  it('facets should have the correct function selectors', async () => {
    let selectors = getSelectors(diamondCutFacet);
    result = await diamondLoupeFacet.facetFunctionSelectors(
      diamondCutFacetAddress,
    );
    assert.sameMembers(result, selectors);

    selectors = getSelectors(diamondLoupeFacet);
    result = await diamondLoupeFacet.facetFunctionSelectors(
      diamondLoupeFacetAddress,
    );
    assert.sameMembers(result, selectors);

    selectors = getSelectors(ownershipFacet);
    result = await diamondLoupeFacet.facetFunctionSelectors(
      ownershipFacetAddress,
    );
    assert.sameMembers(result, selectors);

    selectors = getSelectors(tradingManagementStorageFacet);
    result = await diamondLoupeFacet.facetFunctionSelectors(
      tradingManagementStorageFacetAddress,
    );
    assert.sameMembers(result, selectors);
  });

  it('selectors should be correctly associated with their respective facets', async () => {
    assert.equal(
      diamondCutFacetAddress,
      await diamondLoupeFacet.facetAddress('0x1f931c1c'), // DiamondCutFacet's function selector
    );
    assert.equal(
      diamondLoupeFacetAddress,
      await diamondLoupeFacet.facetAddress('0xcdffacc6'), // DiamondLoupeFacet's facets()
    );
    assert.equal(
      diamondLoupeFacetAddress,
      await diamondLoupeFacet.facetAddress('0x01ffc9a7'), // DiamondLoupeFacet's supportsInterface
    );
    assert.equal(
      ownershipFacetAddress,
      await diamondLoupeFacet.facetAddress('0xf2fde38b'), // OwnershipFacet's transferOwnership
    );
    assert.equal(
      tradingManagementStorageFacetAddress,
      await diamondLoupeFacet.facetAddress('0xf05a7662'), // TradingManagementStorageFacet's setBourseContractAddress
    );
    assert.equal(
      tradingManagementExecutorFacetAddress,
      await diamondLoupeFacet.facetAddress('0xe9ee92fc'), // TradingManagementExecutorFacet's setBourseContractAddress
    );
  });

  it('should add Test1Facet functions to the Diamond', async () => {
    const Test1Facet = await ethers.getContractFactory('Test1Facet');
    const test1Facet = await Test1Facet.deploy();
    await test1Facet.deployed();
    test1FacetAddress = test1Facet.address;
    addresses.push(test1FacetAddress);

    const selectors = removeSelectors(getSelectors(test1Facet), [
      'supportsInterface(bytes4)',
    ]);

    tx = await diamondCutFacet.diamondCut(
      [
        {
          facetAddress: test1FacetAddress,
          action: FacetCutAction.Add,
          functionSelectors: selectors,
        },
      ],
      ethers.constants.AddressZero,
      '0x',
      { gasLimit: 800000 },
    );
    receipt = await tx.wait();

    if (!receipt.status) {
      throw Error(`Diamond upgrade failed: ${tx.hash}`);
    }

    result = await diamondLoupeFacet.facetFunctionSelectors(test1FacetAddress);
    assert.sameMembers(result, selectors);
  });

  it('should call a function from Test1Facet through the Diamond', async () => {
    const test1Facet = await ethers.getContractAt('Test1Facet', DiamondAddress);
    await test1Facet.test1Func10();
    // Add assertions here to verify state changes or events if applicable
  });

  it('should replace the supportsInterface function with Test1Facet implementation', async () => {
    const t1facet = await ethers.getContractFactory('Test1Facet');
    const selectors = getSelectors(t1facet).get(['supportsInterface(bytes4)']);
    const testFacetAddress = test1FacetAddress;

    tx = await diamondCutFacet.diamondCut(
      [
        {
          facetAddress: testFacetAddress,
          action: FacetCutAction.Replace,
          functionSelectors: selectors,
        },
      ],
      ethers.constants.AddressZero,
      '0x',
      { gasLimit: 800000 },
    );
    receipt = await tx.wait();

    if (!receipt.status) {
      throw Error(`Diamond upgrade failed: ${tx.hash}`);
    }

    result = await diamondLoupeFacet.facetFunctionSelectors(testFacetAddress);
    assert.sameMembers(result, getSelectors(t1facet));
  });

  it('should add Test2Facet functions to the Diamond', async () => {
    const Test2Facet = await ethers.getContractFactory('Test2Facet');
    const test2Facet = await Test2Facet.deploy();
    await test2Facet.deployed();
    test2FacetAddress = test2Facet.address;
    addresses.push(test2FacetAddress);

    const selectors = getSelectors(test2Facet);
    tx = await diamondCutFacet.diamondCut(
      [
        {
          facetAddress: test2FacetAddress,
          action: FacetCutAction.Add,
          functionSelectors: selectors,
        },
      ],
      ethers.constants.AddressZero,
      '0x',
      { gasLimit: 800000 },
    );
    receipt = await tx.wait();

    if (!receipt.status) {
      throw Error(`Diamond upgrade failed: ${tx.hash}`);
    }

    result = await diamondLoupeFacet.facetFunctionSelectors(test2FacetAddress);
    expect(result.length).equals(selectors.length);
    assert.sameMembers(result, selectors);
  });

  it('should have the correct total number of functions from Test1Facet and Test2Facet combined', async () => {
    const test1Selectors = getSelectors(
      await ethers.getContractAt('Test1Facet', test1FacetAddress),
    );
    const test2Selectors = getSelectors(
      await ethers.getContractAt('Test2Facet', test2FacetAddress),
    );
    const totalExpectedSelectors =
      test1Selectors.length + test2Selectors.length;

    const result1 =
      await diamondLoupeFacet.facetFunctionSelectors(test1FacetAddress);
    const result2 =
      await diamondLoupeFacet.facetFunctionSelectors(test2FacetAddress);
    const totalSelectors = result1.length + result2.length;

    expect(totalSelectors).equals(totalExpectedSelectors);
  });

  it('should remove specific functions from Test2Facet, leaving only 5 functions', async () => {
    const test2Facet = await ethers.getContractAt('Test2Facet', DiamondAddress);

    const functionsToKeep = [
      'test2Func1()',
      'test2Func5()',
      'test2Func6()',
      'test2Func19()',
      'test2Func20()',
    ];

    const selectorsToRemove = removeSelectors(
      getSelectors(test2Facet),
      functionsToKeep,
    );

    tx = await diamondCutFacet.diamondCut(
      [
        {
          facetAddress: ethers.constants.AddressZero,
          action: FacetCutAction.Remove,
          functionSelectors: selectorsToRemove,
        },
      ],
      ethers.constants.AddressZero,
      '0x',
      { gasLimit: 800000 },
    );
    receipt = await tx.wait();

    if (!receipt.status) {
      throw Error(`Diamond upgrade failed: ${tx.hash}`);
    }

    const remainingSelectors =
      await diamondLoupeFacet.facetFunctionSelectors(test2FacetAddress);
    const expectedSelectors = getSelectors(test2Facet).get(functionsToKeep);

    assert.sameMembers(remainingSelectors, expectedSelectors);
  });

  it('should remove specific functions from Test1Facet, leaving only 3 functions', async () => {
    const test1Facet = await ethers.getContractAt('Test1Facet', DiamondAddress);

    const functionsToKeep = ['test1Func2()', 'test1Func11()', 'test1Func12()'];

    const selectorsToRemove = removeSelectors(
      getSelectors(test1Facet),
      functionsToKeep,
    );

    tx = await diamondCutFacet.diamondCut(
      [
        {
          facetAddress: ethers.constants.AddressZero,
          action: FacetCutAction.Remove,
          functionSelectors: selectorsToRemove,
        },
      ],
      ethers.constants.AddressZero,
      '0x',
      { gasLimit: 800000 },
    );
    receipt = await tx.wait();

    if (!receipt.status) {
      throw Error(`Diamond upgrade failed: ${tx.hash}`);
    }

    const remainingSelectors =
      await diamondLoupeFacet.facetFunctionSelectors(test1FacetAddress);
    const expectedSelectors = getSelectors(test1Facet).get(functionsToKeep);

    assert.sameMembers(remainingSelectors, expectedSelectors);
  });

  it('should have a total of 8 functions from Test1Facet and Test2Facet after removals', async () => {
    const result1 =
      await diamondLoupeFacet.facetFunctionSelectors(test1FacetAddress);
    const result2 =
      await diamondLoupeFacet.facetFunctionSelectors(test2FacetAddress);
    expect(result1.length + result2.length).equals(8);
  });

  it("should remove all functions and facets except 'diamondCut' and 'facets'", async () => {
    let selectors = [];
    let facets = await diamondLoupeFacet.facets();

    for (let i = 0; i < facets.length; i++) {
      selectors.push(...facets[i].functionSelectors);
    }

    selectors = removeSelectors(selectors, [
      'facets()',
      'diamondCut(tuple(address,uint8,bytes4[])[],address,bytes)',
    ]);

    tx = await diamondCutFacet.diamondCut(
      [
        {
          facetAddress: ethers.constants.AddressZero,
          action: FacetCutAction.Remove,
          functionSelectors: selectors,
        },
      ],
      ethers.constants.AddressZero,
      '0x',
      { gasLimit: 800000 },
    );
    receipt = await tx.wait();

    if (!receipt.status) {
      throw Error(`Diamond upgrade failed: ${tx.hash}`);
    }

    facets = await diamondLoupeFacet.facets();
    assert.equal(facets.length, 2);
    assert.equal(facets[0].facetAddress, diamondCutFacetAddress);
    assert.sameMembers(facets[0].functionSelectors, ['0x1f931c1c']); // diamondCut function
    assert.equal(facets[1].facetAddress, diamondLoupeFacetAddress);
    assert.sameMembers(facets[1].functionSelectors, ['0x7a0ed627']); // facets function
  });

  it('should add back most functions and facets to the Diamond', async () => {
    const diamondLoupeFacetSelectors = removeSelectors(
      getSelectors(diamondLoupeFacet),
      ['supportsInterface(bytes4)'],
    );

    const test1Facet = await ethers.getContractFactory('Test1Facet');
    const test2Facet = await ethers.getContractFactory('Test2Facet');

    const cut = [
      {
        facetAddress: diamondLoupeFacetAddress,
        action: FacetCutAction.Add,
        functionSelectors: removeSelectors(diamondLoupeFacetSelectors, [
          'facets()',
        ]),
      },
      {
        facetAddress: ownershipFacetAddress,
        action: FacetCutAction.Add,
        functionSelectors: getSelectors(ownershipFacet),
      },
      {
        facetAddress: test1FacetAddress,
        action: FacetCutAction.Add,
        functionSelectors: getSelectors(test1Facet),
      },
      {
        facetAddress: test2FacetAddress,
        action: FacetCutAction.Add,
        functionSelectors: getSelectors(test2Facet),
      },
      {
        facetAddress: tradingManagementStorageFacetAddress,
        action: FacetCutAction.Add,
        functionSelectors: getSelectors(tradingManagementStorageFacet),
      },
    ];

    tx = await diamondCutFacet.diamondCut(
      cut,
      ethers.constants.AddressZero,
      '0x',
      { gasLimit: 8000000 },
    );
    receipt = await tx.wait();

    if (!receipt.status) {
      throw Error(`Diamond upgrade failed: ${tx.hash}`);
    }

    const facets: any[] = await diamondLoupeFacet.facets();
    const facetAddresses = await diamondLoupeFacet.facetAddresses();

    assert.equal(facetAddresses.length, 6);
    assert.equal(facets.length, 6);
    assert.sameMembers(facetAddresses, [
      diamondCutFacetAddress,
      diamondLoupeFacetAddress,
      ownershipFacetAddress,
      test1FacetAddress,
      test2FacetAddress,
      tradingManagementStorageFacetAddress,
    ]);

    // Define the expected selectors for DiamondLoupeFacet
    const expectedDiamondLoupeSelectors = removeSelectors(
      getSelectors(diamondLoupeFacet),
      ['supportsInterface(bytes4)'],
    );

    assert.equal(
      facets[findAddressPositionInFacets(diamondCutFacetAddress, facets)]
        .facetAddress,
      diamondCutFacetAddress,
      'DiamondCutFacet address mismatch',
    );
    assert.equal(
      facets[findAddressPositionInFacets(diamondLoupeFacetAddress, facets)]
        .facetAddress,
      diamondLoupeFacetAddress,
      'DiamondLoupeFacet address mismatch',
    );
    assert.equal(
      facets[findAddressPositionInFacets(ownershipFacetAddress, facets)]
        .facetAddress,
      ownershipFacetAddress,
      'OwnershipFacet address mismatch',
    );
    assert.equal(
      facets[findAddressPositionInFacets(test1FacetAddress, facets)]
        .facetAddress,
      test1FacetAddress,
      'Test1Facet address mismatch',
    );
    assert.equal(
      facets[findAddressPositionInFacets(test2FacetAddress, facets)]
        .facetAddress,
      test2FacetAddress,
      'Test2Facet address mismatch',
    );
    assert.equal(
      facets[
        findAddressPositionInFacets(
          tradingManagementStorageFacetAddress,
          facets,
        )
      ].facetAddress,
      tradingManagementStorageFacetAddress,
      'TradingManagementStorageFacet address mismatch',
    );

    // Update the assertions to compare with the expected selectors
    assert.sameMembers(
      facets[findAddressPositionInFacets(diamondCutFacetAddress, facets)]
        .functionSelectors,
      getSelectors(diamondCutFacet),
    );
    assert.sameMembers(
      facets[findAddressPositionInFacets(diamondLoupeFacetAddress, facets)]
        .functionSelectors,
      expectedDiamondLoupeSelectors,
    );
    assert.sameMembers(
      facets[findAddressPositionInFacets(ownershipFacetAddress, facets)]
        .functionSelectors,
      getSelectors(ownershipFacet),
    );
    assert.sameMembers(
      facets[findAddressPositionInFacets(test1FacetAddress, facets)]
        .functionSelectors,
      getSelectors(test1Facet),
    );
    assert.sameMembers(
      facets[findAddressPositionInFacets(test2FacetAddress, facets)]
        .functionSelectors,
      getSelectors(test2Facet),
    );
    assert.sameMembers(
      facets[
        findAddressPositionInFacets(
          tradingManagementStorageFacetAddress,
          facets,
        )
      ].functionSelectors,
      getSelectors(tradingManagementStorageFacet),
    );
  });

  it('should transfer ownership of the Diamond', async () => {
    const currentOwner = await ownershipFacet.owner();
    const accounts = await ethers.getSigners();
    const newOwner = accounts[1].address;

    tx = await ownershipFacet.transferOwnership(newOwner);
    receipt = await tx.wait();

    const updatedOwner = await ownershipFacet.owner();
    assert.equal(updatedOwner, newOwner, 'Ownership transfer failed');

    tx = await ownershipFacet
      .connect(accounts[1])
      .transferOwnership(currentOwner);
    receipt = await tx.wait();

    const finalOwner = await ownershipFacet.owner();
    assert.equal(finalOwner, currentOwner, 'Ownership transfer back failed');
  });

  it('should correctly report supported interfaces', async () => {
    const erc165 = await ethers.getContractAt(
      '@openzeppelin/contracts/utils/introspection/IERC165.sol:IERC165',
      DiamondAddress,
    );

    const supportsIERC165 = await erc165.supportsInterface('0x01ffc9a7');
    assert.isTrue(supportsIERC165, 'Diamond does not support IERC165');

    const supportsRandomInterface =
      await erc165.supportsInterface('0xffffffff');
    assert.isFalse(
      supportsRandomInterface,
      'Diamond incorrectly supports random interface',
    );
  });
});
